"""The main screen for the game"""

import random
import math
import pygame
import os
import copy
import time

import serge.actor
import serge.visual
import serge.events
import serge.common
import serge.blocks.utils
import serge.blocks.visualblocks
import serge.blocks.behaviours
import serge.blocks.actors

from theme import G, theme
import common 
import board
import mainmenu
import ai
import enemy
import turret

class MainScreen(serge.blocks.actors.ScreenActor):
    """The logic for the main screen"""
    
    def __init__(self, options, level=None, direction=board.FORWARD):
        """Initialise the screen"""
        super(MainScreen, self).__init__('item', 'main-screen')
        self.options = options
        self.current_level = level if level is not None else G('start-level')
        self.waves = self.wave = None
        self.state = common.S_GAME_START
        self.direction = direction
        
    def addedToWorld(self, world):
        """We were added to the world"""
        super(MainScreen, self).addedToWorld(world)
        self.initialiseLevel()
        
    def initialiseLevel(self):
        """Initialise the level"""
        #
        # The behaviour manager
        self.manager = serge.blocks.behaviours.BehaviourManager('behaviours', 'behaviours')
        self.world.addActor(self.manager)
        world = self.world
        self.level_start_time = time.time()
        #
        level_name = 'level-%d' % self.current_level
        #
        self.game_board = board.GameBoard.fromXMLFile(
            os.path.join('levels', 'level-%d.tmx' % self.current_level), self.direction)
        self.board = board.BoardActor('board', 'board', self.game_board, self, 
            G('board-cell-size'), G('background', level_name), G('show-tiles', level_name))
        serge.blocks.utils.addActorToWorld(world, self.board, layer_name='board', 
            center_position=G('board-position'))
        #
        # Foreground
        fg = G('foreground', level_name)
        if fg:
            self.fg = serge.blocks.utils.addSpriteActorToWorld(world, 'fg', 'fg', fg, layer_name='foreground',
                center_position=G('board-position'))
        self.pain = serge.blocks.utils.addSpriteActorToWorld(world, 'pain', 'pain', 'pain', layer_name='foreground',
            center_position=G('board-position'))
        self.pain.active = False
        self.pain_active = time.time()
        #
        level_name = 'level-%d' % self.current_level
        self.waves = copy.copy(G('waves', level_name))
        self.menu = serge.blocks.utils.addActorToWorld(world, mainmenu.MainMenu(self.board.layout, self.board))
        self.menu.setCash(G('initial-cash', level_name))
        #
        # Achievements
        self.achievements = serge.blocks.achievements.getManager()
        serge.blocks.achievements.addAchievementsBannerToWorld(world, 'front-ui', 'ui-background', theme, self.manager)
        #
        # Game/Level over UI elements
        self.main_title = serge.blocks.utils.addActorToWorld(world,
            serge.blocks.actors.StringText('title', 'main-title', 'Welcome', format='%s', 
                colour=G('main-title-colour'), font_size=G('main-title-size')), 
            layer_name='front-ui', center_position=G('main-title-position'))
        self.sub_title = serge.blocks.utils.addVisualActorToWorld(world, 'title', 'sub-title', 
                serge.blocks.visualblocks.RectangleText('Start', G('sub-title-colour'),
                G('sub-title-button-size'), G('sub-title-button-colour'), font_size=G('sub-title-size'),
                stroke_width=G('sub-title-button-stroke'), stroke_colour=G('sub-title-button-stroke-colour')), 
            layer_name='front-ui', center_position=G('sub-title-position'))
        self.sub_title.linkEvent(serge.events.E_LEFT_CLICK, self.titleClick)
        self.sub_title2 = serge.blocks.utils.addVisualActorToWorld(world, 'title', 'sub-title-2', 
                serge.blocks.visualblocks.RectangleText('Replay Level', G('sub-title-colour'),
                G('sub-title-button-size'), G('sub-title-button-colour'), font_size=G('sub-title-size'),
                stroke_width=G('sub-title-button-stroke'), stroke_colour=G('sub-title-button-stroke-colour')), 
            layer_name='front-ui', center_position=G('sub-title-2-position'))
        self.sub_title2.linkEvent(serge.events.E_LEFT_CLICK, self.title2Click)
        self.sub_title2.active = False
        #
        self.enemy_timer = self.manager.assignBehaviour(self, 
            serge.blocks.behaviours.TimedCallback(G('enemy-interval'), self.nextEnemy), 'enemy')
        #
        # The AI
        self.ai = serge.blocks.utils.addActorToWorld(world, ai.AI('ai', 'ai', self.board, 
            G('initial-intelligence', level_name)))
        #
        # Test turrets
        if self.options.include:
            for turret_type, (x, y) in G('test-turrets', level_name):
                self.board.placeTurret(turret_type, (x, y))
        #
        # Hints
        self.hints = []
        for time_start, time_stop, sprite, position in G('hints', level_name):
            hint = serge.blocks.utils.addSpriteActorToWorld(world, 'hint', sprite, sprite, layer_name='hints',
                center_position=position)
            hint.active = (time_start == 0)
            self.hints.append((time_start, time_stop, hint))
        #
        # If we are starting for the first time then pause things
        if self.state == common.S_GAME_START:
            self.enemy_timer.pause()
            self.menu.setActive(False)
        else:
            self.main_title.active = self.sub_title.active = False
        #
        # Some stuff to help while cheating / debugging
        if self.options.cheat:
            self.cheat_goal = serge.blocks.utils.addActorToWorld(world,
                serge.blocks.actors.NumericText('cheat', 'cheat-goal', 'Goal: %d', 
                    G('cheat-colour'), font_size=G('cheat-size'), value=0),
                layer_name='ui', center_position=G('cheat-goal-position'))
            self.cheat_smart = serge.blocks.utils.addActorToWorld(world,
                serge.blocks.actors.NumericText('cheat', 'cheat-smart', 'Smart: %d%%', 
                    G('cheat-colour'), font_size=G('cheat-size'), value=0),
                layer_name='ui', center_position=G('cheat-smart-position'))
            self.cheat_fps = serge.blocks.utils.addActorToWorld(world,
                serge.blocks.actors.NumericText('cheat', 'cheat-fps', 'FPS: %5.2f', 
                    G('cheat-colour'), font_size=G('cheat-size'), value=0),
                layer_name='ui', center_position=G('cheat-fps-position'))
        #

                   
    def nextWave(self):
        """Next wave of enemies"""
        if self.waves:
            self.wave = list(self.waves.pop(0))
        else:
            if self.enemy_timer.isRunning():
                self.enemy_timer.pause()
            self.broadcaster.processEvent((common.E_WAVES_COMPLETE, self))

    def updateActor(self, interval, world):
        """Update this actor"""
        super(MainScreen, self).updateActor(interval, world)
        #
        # Debugging        
        if self.options.debug and self.keyboard.isClicked(pygame.K_F4):
            # Open a shell
            from IPython.Shell import IPythonShellEmbed  
            ipshell = IPythonShellEmbed(['-gthread'], 'Game Interactive Shell', 'Left Shell', user_ns=locals())  
            ipshell() 
        #
        # Hints
        if self.state == common.S_PLAYING:
            level_run_time = time.time()-self.level_start_time
            for start_time, stop_time, hint in self.hints:
                hint.active = start_time < level_run_time < stop_time
        #
        if self.state == common.S_PLAYING and (not self.wave and not self.waves and not self.world.findActorsByTag('enemy')):
            #
            # Record the achivements
            self.achievements.makeReport('level-complete', 
                level=self.current_level, health=self.menu.getHealth(), time=level_run_time, cash=self.menu.getCash())
            #
            if not theme.hasTheme('level-%d' % (self.current_level+1)):
                self.gameFinished()
            else:
                self.log.info('All enemies destroyed')
                self.main_title.active = self.sub_title.active = True
                self.main_title.value = 'Level Over'
                self.sub_title.visual.text_visual.setText('Next Level')
                self.menu.setActive(False)
                self.state = common.S_LEVEL_OVER
        #
        # Record the achivements
        if self.state == common.S_PLAYING:
            self.achievements.makeReport('playing', 
                time=level_run_time, level=self.current_level, cash=self.menu.getCash())
        #
        if self.options.cheat:
            self.checkGoalChange()
            self.checkSmartChange()
            self.checkFinish()
            self.cheat_goal.value = self.board.goal_index
            self.cheat_smart.value = self.board.board.dps_length_equivalence/G('dps-length-equivalence')*100.0
            self.cheat_fps.value = self.engine.getStats().average_frame_rate
            if self.keyboard.isClicked(pygame.K_p):
                self.goalTakeDamage(1)
        #
        # Pausing
        if self.state == common.S_PLAYING and self.keyboard.isClicked(pygame.K_ESCAPE):
            self.escapeClicked()
        #
        # Clear the pain indicator
        if self.pain.active:
            if time.time() - self.pain_active >= G('pain-active-time'):
                self.pain.active = False
            
    def nextEnemy(self, world, actor, interval):
        """Release the next enemy"""
        if not self.state == common.S_PAUSED:
            self.log.debug('nextEnemy called')
            #
            # Do waves
            if not self.wave:
                self.nextWave()
            else:
                enemy_type = self.wave.pop(0)
                if enemy_type != '-':
                    enemy = self.board.addEnemy('enemy-%s' % enemy_type)
        
    def goalTakeDamage(self, damage):
        """Damage is inflicted on the goal"""
        self.log.info('Damage %s inflicted on the goal' % damage)
        self.menu.takeDamage(damage)
        self.pain.active = True
        self.pain_active = time.time()
        serge.sound.Sounds.play('damage')
        if self.menu.isDestroyed():
            self.log.info('The base is destroyed')
            self.gameOver()
            
    def gameOver(self):
        """The game ended"""
        if self.enemy_timer.isRunning():
            self.enemy_timer.pause()
        self.main_title.active = self.sub_title.active = self.sub_title2.active = True
        self.main_title.value = 'Game Over'
        self.sub_title.visual.text_visual.setText('Restart Game')
        self.sub_title2.visual.text_visual.setText('Restart Level')
        self.menu.setActive(False)
        self.state = common.S_GAME_OVER
            
    def destroyedEnemy(self, enemy):
        """Destroyed an enemy"""
        self.menu.addScore(enemy.score, enemy.cash)
        
    def titleClick(self, obj, arg):
        """The title was clicked"""
        self.log.info('Title clicked')
        if self.state == common.S_GAME_START:
            self.level_start_time = time.time()
            self.menu.setActive(True)
            self.enemy_timer.restart()
            self.main_title.active = self.sub_title.active = self.sub_title2.active = False
            self.state = common.S_PLAYING
        elif self.state in (common.S_GAME_OVER, common.S_FINISHED):
            self.clearActors()
            self.engine.setCurrentWorldByName('start-screen')
        elif self.state == common.S_LEVEL_OVER:
            self.engine.getWorld('start-screen').findActorByName('start-screen').completeLevel(self.current_level)
            unlocks = G('unlocks', 'level-%d' % self.current_level)
            if unlocks == 'next':
                unlocks = [self.current_level+1]
            for level in unlocks:
                self.engine.getWorld('start-screen').findActorByName('start-screen').unlockLevel(level)
            self.current_level += 1
            self.waves = self.wave = None
            self.clearActors()
            self.engine.setCurrentWorldByName('start-screen')
        elif self.state == common.S_PAUSED:
            self.engine.stop()

    def title2Click(self, obj, arg):
        """Subtitle 2 was clicked"""
        if self.state == common.S_PAUSED:
            self.menu.setActive(True)
            self.main_title.active = self.sub_title.active = self.sub_title2.active = False
            enemy.Enemy.paused = False
            turret.Turret.paused = False
            self.state = common.S_PLAYING
        else:
            self.waves = self.wave = None
            self.clearActors()
            self.state = common.S_PLAYING
            self.world.addActor(self)
        
            
    def gameFinished(self):
        """The game has been completed"""
        self.log.info('All level completed')
        self.engine.getWorld('start-screen').findActorByName('start-screen').completeLevel(self.current_level)
        self.main_title.active = self.sub_title.active = True
        self.main_title.value = 'You beat the game!'
        self.sub_title.value = 'Click to Replay'
        self.menu.setActive(False)
        self.state = common.S_FINISHED
        
    def checkGoalChange(self):
        """Check if we should change goals"""
        if self.keyboard.isClicked(pygame.K_F1):
            self.board.setGoal(1)
        elif self.keyboard.isClicked(pygame.K_F2):
            self.board.setGoal(2)
        elif self.keyboard.isClicked(pygame.K_F3):
            self.board.setGoal(3)

    def checkSmartChange(self):
        """Check if we should change smartness"""
        if self.keyboard.isClicked(pygame.K_F5):
            self.board.setSmartness(0.0)
        elif self.keyboard.isClicked(pygame.K_F6):
            self.board.setSmartness(20.0)
        elif self.keyboard.isClicked(pygame.K_F7):
            self.board.setSmartness(40.0)
        elif self.keyboard.isClicked(pygame.K_F8):
            self.board.setSmartness(60.0)
        elif self.keyboard.isClicked(pygame.K_F9):
            self.board.setSmartness(80.0)
        elif self.keyboard.isClicked(pygame.K_F10):
            self.board.setSmartness(100.0)

    def checkFinish(self):
        """Check if we should finish the level now"""
        if self.keyboard.isClicked(pygame.K_f):
            self.wave = self.waves = None
            self.world.clearActorsWithTags(['enemy'])
              
    def escapeClicked(self):
        """The escape key was clicked"""
        if self.state == common.S_PAUSED:
            self.engine.stop()
        self.main_title.active = self.sub_title.active = self.sub_title2.active = True
        self.main_title.value = 'Paused'
        self.sub_title.visual.text_visual.setText('Quit')
        self.sub_title2.visual.text_visual.setText('Resume')
        self.state = common.S_PAUSED
        self.menu.setActive(False)
        enemy.Enemy.paused = True
        turret.Turret.paused = True
        self.menu.selected = None
                  
    def clearActors(self):
        """Clear actors from the world"""
        self.world.clearActorsExceptTags(['mute-button'])
                       
def main(options, level=None, direction=board.FORWARD):
    """Create the main logic"""
    #
    # The screen actor
    s = MainScreen(options, level, direction)
    world = serge.engine.CurrentEngine().getWorld('main-screen')
    world.addActor(s)
    #
    # Screenshots
    if options.screenshot:
        manager = world.findActorByName('behaviours')
        manager.assignBehaviour(None, 
            serge.blocks.behaviours.SnapshotOnKey(key=pygame.K_s, size=G('screenshot-size')
                , overwrite=False, location='screenshots'), 'screenshots')
    #
    return s
